{ ******************************************************************************
   PACIFICA DATE-TIME DATA TYPES AND ROUTINES UNIT

   Originally created in 1993.
   Updated and adapted to Free Pascal, March 2011.
   Latest Update: 2017-10-20: Updated pToC()

   Dates and times are based on elapsed integer seconds from 00:00 hours,
   January 1, 1800.

   The algorithms factor in:
   - Leap years
   - 'Century' non-leap years; i.e. 1800, 1900, 2100, 2200 ...
     - 'Century' leap years; i.e. 2000, 2400, 2800, ...
       (i.e. divisible by 400)
   They do not factor in:
   - Decamillennial (sp?) non-leap years; i.e. 40000, 80000, ...
   - Leap seconds as inserted into UTC.
   - The Julian Calendar; thus origin-1800.  You Russians, well...  :(
   Times are recorded in seconds from midnight.
   A pDateTime value includes the datevalue, multiplied by 86,400,
   plus the timevalue.
   A value of 0 is an invalid datevalue (January 0, 1800)
}


unit PacDateTime;

{$mode objfpc}{$H+}

interface

uses
  Classes, SysUtils, PacString;

type
  pDateTime = Int64;
  pDate = pDateTime;
  pTime = pDateTime;
  pDateL = record
    YY : word;
    MM,DD : byte;
  end;
  pTimeL = record
    H,M,S : byte;
  end;
  pDateTimeL = record
    D : pDateL;
    T : pTimeL;
  end;

var
   Dim,Dtm : array[ 1..12 ] of word;      { Days/Month, Days (prior) To Month}
   Nom : array[ 1..12 ] of string[ 9 ];   { Names of Months }
   Wkd : array[ 1..7 ] of string[ 9 ];    { Weekday names }
   DateSep : char;                        { Date separator character used for date text routines.
                                            Initialized in proc ptd_init() }

function TimeBit( W : word ) : string;
function CDateX : string;
function CDate : string;
function LiteralDate( AD : longint ) : string;
function CTimeX : string;
function CTime : string;
function LeapYear( Yr : word ) : boolean;
function DaysInMonth( Yr : word; Mo : byte ) : byte;
function DaysInYear( Yr : word ) : word;
function DaysElapsed( AD : longint ) : word;
function DaysElapsed2( Yr,Mo,Dy : word ) : word;
function DaysLeft( AD : longint ) : word;
function DaysLeft2( Yr,Mo,Dy : word ) : word;
function DayOfYear( AD : longint ) : word;
function DayOfYear2( Yr,Mo,Dy : word ) : word;
procedure UnpackSeconds( TS : longint; var D,H,M,S : byte );
function PackSeconds( D,H,M,S : byte ) : longint;
procedure AddDays( var Y,M,D : word; Days : word );
function ValidDate( CD : string ) : boolean;
function ValidTime( CT : string ) : boolean;
procedure FillZeroes( var S : string );
function CtoD( CD : string ) : longint;
procedure CtoT( CT : string; var H,M : word );
function CtoT(CT : String) : longint;
function DtoC( AD : longint ) : String;
function DtoC8( AD : longint ) : String;
function TtoC( H,M : byte ) : String;
function AbsDate( Y,M,D : word ) : longint;
function DaysToMonth( Y,M : word ) : word;
procedure UnAbsDate( A : longint; var Y : word; var M,D : byte );
function DayOfWeek( AD : longint ) : byte;
function MonthName( M : byte ) : string;
function pAbsDate( Y : word; M,D : byte ) : pDate;
procedure pDecodeDate( A : pDate; var Y : word; var M,D : byte );
procedure pEncodeDate( Y : word; M,D : byte; var A : pDate );
function pDayOfYear( D : pDate ) : word;
function pDayOfYear2( Yr : word; Mo,Dy : byte ) : word;
procedure pD2DL( D : pDate; var DL : pDateL );
function pCtoD( CD : string ) : pDate;
function pCtoT( CT : string ) : pTime;
function pCtoMMM( C : string ) : integer;
function pMMMtoC( M : integer ) : string;
procedure pCtoMD( CD : string; var M,D : byte );
function pDtoC( D : pDate ) : String;
function pTtoC( T : pTime ) : string;
function pDayOfWeek( D : pDate ) : byte;
function WeekdayName( D : byte ) : string;
function ShortWeekdayName( D : byte ) : string;
function ShortMonthName( M : byte ) : string;


implementation


{
   Takes a numeric value, creates a two-digit string and adds a
   leading zero if necessary.  Useful for creating time and date
   strings.
   Ie:  TimeBit( 3 )  returns '03'
        TimeBit( 10 ) returns '10'
}
function TimeBit( W : word ) : string;
var Res : string;
begin
   Str( W:2,Res );
   if Res[ 1 ] = ' ' then Res[ 1 ] := '0';
   TimeBit := Res;
end;


{
   Returns the current date in the format YYYY.MM.DD
}
function CDateX : string;
var
   Res : string;
   YY,MM,DD : word;
begin
   DecodeDate( Now(),YY,MM,DD );
   Str( YY:4,Res );
   for YY := 1 to 4 do if Res[ YY ] = ' ' then Res[ YY ] := '0';
   Res := Res+DateSep+TimeBit( MM )+DateSep+TimeBit( DD );
   CDateX := Res;
end;


{
   Returns the current date in the format YY.MM.DD
}
function CDate : string;
var Res : string;
begin
   Res := CDateX;
   Delete( Res,1,2 );
   CDate := Res;
end;


{
   Translates the specified date to the format Month Day, Year
   ie:  January 27, 1992
}
function LiteralDate( AD : longint ) : string;
var
   Temp1,Temp2 : string;
   Yr : word;
   Mo,Dy : byte;
begin
   UnAbsDate( AD,Yr,Mo,Dy );
   Temp1 := Nom[ Mo ]+' ';
   Temp2 := AllTrim( N2S( Dy ));
   Temp1 := Temp1+Temp2+', ';
   Temp2 := AllTrim( N2S( Yr ));
   Temp1 := Temp1+Temp2;
   LiteralDate := Temp1;
end;


{
   Returns the current time as "HH:MM:SS"
}
function CTimeX : string;
var
   H,M,S,S1000 : word;
begin
   DecodeTime( Now,H,M,S,S1000 );
   CTimeX := TimeBit( H )+':'+TimeBit( M )+':'+TimeBit( S );
end;


{
   returns the current time as "HH:MM"
}
function CTime : string;
var Res : string;
begin
   Res := CTimeX;
   Delete( Res,6,3 );
   CTime := Res;
end;


{
   Returns TRUE if the specified year is a leap year
}
function LeapYear( Yr : word ) : boolean;
begin
   if ((Yr mod 4) = 0) or ((Yr mod 400) = 0) then
      LeapYear := True else
      LeapYear := False;
end;


{
   Returns the number of days in the month, accounting for leap years
}
function DaysInMonth( Yr : word; Mo : byte ) : byte;
var Temp : byte;
begin
   Temp := Dim[ Mo ];
   if Mo = 2 then
      if LeapYear( Yr ) then Inc( Temp );
   DaysInMonth := Temp;
end;


{
   Returns the number of days in the specified year, taking leap years
   into account
}
function DaysInYear( Yr : word ) : word;
begin
   if LeapYear( Yr ) then
      DaysInYear := 366 else
      DaysInYear := 365;
end;


{
   Returns the number of days elapsed to the specified date, not counting
   that date
}
function DaysElapsed( AD : longint ) : word;
var Temp,Yr : word; Mo,Dy : byte;
begin
   UnAbsDate( AD,Yr,Mo,Dy );
   Temp := Dtm[ Mo ]+Pred( Dy );
   if (LeapYear( Yr )) and (Mo > 2) then Inc( Temp );
   DaysElapsed := Temp;
end;


function DaysElapsed2( Yr,Mo,Dy : word ) : word;
var Temp : word;
begin
   Temp := Dtm[ Mo ]+Pred( Dy );
   if (LeapYear( Yr )) and (Mo > 2) then Inc( Temp );
   DaysElapsed2 := Temp;
end;


{
   Calculate the number of days left in the year, not counting the
   specified day
}
function DaysLeft( AD : longint ) : word;
var Temp,Yr : word; Mo,Dy : byte;
begin
   UnAbsDate( AD,Yr,Mo,Dy );
   Temp := Dtm[ Mo ]+Pred( Dy );
   if LeapYear( Yr ) and (Mo > 2) then Inc( Temp );
   Daysleft := Pred( DaysInYear( Yr )-Temp );
end;


function DaysLeft2( Yr,Mo,Dy : word ) : word;
var Temp : word;
begin
   Temp := Dtm[ Mo ]+Pred( Dy );
   if LeapYear( Yr ) and (Mo > 2) then Inc( Temp );
   Daysleft2 := Pred( DaysInYear( Yr )-Temp );
end;


{
   Calculates the day number within the year ( := Succ( DaysElapsed()) )
}
function DayOfYear( AD : longint ) : word;
var Temp,Yr : word; Mo,Dy : byte;
begin
   UnAbsDate( AD,Yr,Mo,Dy );
   Temp := Dtm[ Mo ]+Dy;
   if (LeapYear( Yr )) and (Mo > 2) then Inc( Temp );
   DayOfYear := Temp;
end;


function DayOfYear2( Yr,Mo,Dy : word ) : word;
var Temp : word;
begin
   Temp := Dtm[ Mo ]+Dy;
   if (LeapYear( Yr )) and (Mo > 2) then Inc( Temp );
   DayOfYear2 := Temp;
end;


{
   Break-out a number of seconds
      TS:  number of seconds in the time period
      D:  number of days in the time period
      H:  number of hours
      M:  number of minutes
      S:  number of seconds
}
procedure UnpackSeconds( TS : longint; var D,H,M,S : byte );
begin
   S := TS mod 60;
   TS := (TS-S) div 60;
   M := TS mod 60;
   TS := (TS-M) div 60;
   H := TS mod 24;
   TS := (TS-H) div 24;
   D := TS;
end;


{
   Calculate total seconds in a given time period
      D:  number of days in the time period
      H:  number of hours
      M:  number of minutes
      S:  number of seconds
}
function PackSeconds( D,H,M,S : byte ) : longint;
begin
   PackSeconds := (((((longint(D)*longint(24))
                  +longint(H))*longint(60))
                  +longint(M))*longint(60))
                  +longint(S)
end;


{
   Add a number of days to a year value
      Y,M,D:  Year,month,day of starting date
      Days:   Number of days to add
   Updates the Y,M,D values appropriately
}
procedure AddDays( var Y,M,D : word; Days : word );
var OK : boolean;
begin
   D := D+Days;
   OK := False;
   repeat
      if D > DaysInMonth( Y,M ) then begin
         D := D-DaysInMonth( Y,M );
         Inc( M );
         if M > 12 then begin
            M := M-12;
            Inc( Y );
         end;
      end else OK := True;
   until OK;
end;


{
   Determine whether a given date is valid
   NOTE:  Date string passed MUST be eight characters in length!
          Date is prefixed by '19'
}
function ValidDate( CD : string ) : boolean;
var
   Y : word;
   M,D : byte;
begin
   Y := AtoI( SubStr( CD,1,2 ))+1900;
   M := AtoB( SubStr( CD,4,2 ));
   D := AtoB( SubStr( CD,7,2 ));
   if (M < 1) or (M > 12) then ValidDate := FALSE
   else if (D < 1) or (D > DaysInMonth( Y,M ))
   then ValidDate := FALSE
   else ValidDate := TRUE;
end;


{
   Validate a given time
   NOTE:  Time string passed MUST be five characters in length!
          Time MUST be in 24-hour format.
}
function ValidTime( CT : string ) : boolean;
var H,M : byte;
begin
   if Pos( ' ',CT ) <> 0 then ValidTime := FALSE else begin
      H := AtoI( SubStr( CT,1,2 ));
      M := AtoI( SubStr( CT,4,2 ));
      if (H < 24) and (M < 60)
      then ValidTime := TRUE
      else ValidTime := FALSE;
   end;
end;


{
   Convenience:  replaces all space characters in a string with '0' chars.
}
procedure FillZeroes( var S : string );
var Cnt : byte;
begin
   for Cnt := 1 to Length( S ) do
      if S[ Cnt ] = ' ' then S[ Cnt ] := '0';
end;


{
   Converts a date in the format YY.MM.DD or YYYY.MM.DD to its components.
   NOTE:  Date string passed MUST contain valid digits in all positions!
}
function CtoD( CD : string ) : longint;
var
   Cnt : byte;
   Frag : string;
   Y,M,D : word;
begin
   Cnt := Pos( DateSep,CD );
   Frag := LeftStr( CD,Pred( Cnt ));
   Y := AtoI( Frag );
   if Y < 1800 then begin
      if Y < 100 then begin
         if Y > 69 then Y := Y+word(1900) else Y := Y+word(2000);
      end else begin
         Cnt := 100;
         CtoD := 1;
      end;
   end;
   if Cnt <> 100 then begin
      Delete( CD,1,Cnt );
      Frag := SubStr( CD,1,2 );
      M := AtoI( Frag );
      Frag := SubStr( CD,4,2 );
      D := AtoI( Frag );
      CtoD := AbsDate( Y,M,D );
   end;
end;


{
   Converts a time in the format HH:MM to its components.
   NOTE:  Time string passed MUST contain valid digits in all positions!
}
procedure CtoT( CT : string; var H,M : word );
var Frag : string[ 2 ];
begin
   Frag := LeftStr( CT,2 );
   H := AtoI( Frag );
   Frag := RightStr( CT,2 );
   M := AtoI( Frag );
end;

{
   Converts a time in the format HH:MM to a timeval.
   NOTE:  Time string passed MUST contain valid digits in all positions!
}
function CtoT(CT : String) : longint;
var
  Frag : String[2];
  H,M : word;
  R : longint;
begin
     Frag := LeftStr(CT,2);
     H := AtoI(Frag);
     Frag := RightStr( CT,2 );
     M := AtoI( Frag );
     R := ((H*60)+M)*60;
     CtoT := R;
end;

{ Converts a raw time value to a 10-digit string in the format YYYY-MM-YY }
function DtoC( AD : longint ) : String;
var
   Res : String;
   Y : word; M,D : byte;
begin
   UnAbsDate( AD,Y,M,D );
   Res := Cn2s( Y,2 )+DateSep+Cn2s( M,2 )+DateSep+Cn2s( D,2 );
   FillZeroes( Res );
   DtoC := Res;
end;


{ Converts a raw time value to an 8-digit string in the format YYYYMMYY }
function DtoC8( AD : longint ) : String;
var
   Res : String;
   Y : word; M,D : byte;
begin
   UnAbsDate( AD,Y,M,D );
   Res := Cn2s( Y,2 )+Cn2s( M,2 )+Cn2s( D,2 );
   FillZeroes( Res );
   DtoC8 := Res;
end;


function DtoC2( Y,M,D : word ) : String;
var Res : String;
begin
   Res := Cn2s( Y,2 )+DateSep+Cn2s( M,2 )+DateSep+Cn2s( D,2 );
   FillZeroes( Res );
   DtoC2 := Res;
end;


function TtoC( H,M : byte ) : String;
var Res : String;
begin
   Res := TimeBit(H)+':'+TimeBit(M);
   TtoC := Res;
end;


{
   Convert to absolute date (days since 1799.12.31)
   NOTE:  VALID ONLY ON DATES 1800.01.01 AND LATER
          1800.01.01 is Absolute Day #1
}
function AbsDate( Y,M,D : word ) : longint;
var Res : longint;
begin
{ First, calculate the number of days based on 365-day years }
   Res := (longint(Y)-1800)*longint(365);
{ If the year is 1800, result is simply the day of the year }
   if Res = 0 then Res := Res+DayOfYear2( Y,M,D ) else begin
{ Add the number of leap days to the BEGINNING of the year }
      Res := Res+Succ( (Pred( Y )-1800) DIV 4);
{ Subtract leap days SKIPPED in century years }
{ (ie:  1800,1900,2100,2200,2300, etc.) }
      Res := Res-((Pred( Y )-1600) DIV 100);
{ Add leap days in century years DIVISIBLE BY 400 }
      Res := Res+((Pred( Y )-1600) DIV 400);
{ Add days in current year }
      Res := Res+DayOfYear2( Y,M,D );
   end;
   AbsDate := Res;
end;


{
   CALCULATE DAYS PRIOR TO BEGINNING OF MONTH
}
function DaysToMonth( Y,M : word ) : word;
begin
   if (LeapYear( Y )) and (M > 2)
   then DaysToMonth := Succ( Dtm[ M ] )
   else DaysToMonth := Dtm[ M ];
end;


{
   Convert absolute date to Y.M.D
}
procedure UnAbsDate( A : longint; var Y : word; var M,D : byte );
var Rem,Carry,Cnt : word;
begin
{ If we're talking 1801.01.01 or later, have to work out years }
   if A > 365 then begin
{ Calculate number of basic 365-day years elapsed }
      Y := (A DIV word(365))+word(1800);
{ Get the remainder }
{ (Remember:  we haven't accounted for leap years yet) }
      Rem := A-(((longint(Y)-longint(1800))*longint(365)));
{ Add leap days }
      Carry := Succ( (Pred( Y )-1800) DIV 4);
{ Subtract century leap days }
      Carry := Carry-((Pred( Y )-1600) DIV 100);
{ Add 400-year leap days }
      Carry := Carry+((Pred( Y )-1600) DIV 400);
{ if number of extra days is greater than remainder, we're actually }
{ looking at the previous year }
      if Carry >= Rem then begin
{ Count back one year }
         Dec( Y );
{ Add those 365 days to remainder }
         Rem := Rem+365;
{ Calculate leap days }
         Carry := Succ( (Pred( Y )-1800) DIV 4);
{ Subtract century leap days }
         Carry := Carry-((Pred( Y )-1600) DIV 100);
{ Add 400-year leap days }
         Carry := Carry+((Pred( Y )-1600) DIV 400);
      end;
{ Subtract those leap days, giving day number in current year }
      Rem := Rem-Carry;
   end else begin
{ If you can't follow this logic, forget it... }
      Y := 1800;
      Rem := A;
   end;
{ Now calculate the month and day-in-month }
   Cnt := 1;
   while Cnt < 13 do begin
{ Remainder falls within month? }
      if Rem <= DaysInMonth( Y,Cnt ) then begin
{ Cnt is the current month, then. }
         M := Cnt;
{ And remainder is day number }
         D := Rem;
{ And we're outta here! }
         Cnt := 13
      end else begin
{ Subtract days in this month }
         Rem := Rem-DaysInMonth( Y,Cnt );
{ Look ahead to the next month }
         Inc( Cnt );
      end;
   end;
end;


{
   Returns the day of the week for any date after 1699.12.31
   1 = Sunday .. 7 = Saturday
}
function DayOfWeek( AD : longint ) : byte;
begin DayOfWeek := Succ( (AD+3) MOD 7) end;


function MonthName( M : byte ) : string;
begin MonthName := NoM[ M ] end;


{ **************************************************************************** }
{ 2011 ROUTINES BEGIN HERE }

{
   Convert to absolute date (days since 1799.12.31)
   NOTE:  VALID ONLY ON DATES 1800.01.01 AND LATER
          1800.01.01 is Absolute Day #1
}
function pAbsDate( Y : word; M,D : byte ) : pDate;
begin pAbsDate := AbsDate( Y,word(M),word(D) )*int64(86400) end;


procedure pDecodeDate( A : pDate; var Y : word; var M,D : byte );
begin UnAbsDate( A DIV 86400,Y,M,D ) end;


procedure pEncodeDate( Y : word; M,D : byte; var A : pDate );
begin A := AbsDate( Y,M,D )*int64(86400) end;


function pDayOfYear( D : pDate ) : word;
begin pDayOfYear := DayOfYear( D DIV 86400 ) end;


function pDayOfYear2( Yr : word; Mo,Dy : byte ) : word;
begin pDayOfYear2 := DayOfYear2( Yr,word(Mo),word(Dy) ) end;


{ Convert a pDate value into a pDateL record }
procedure pD2DL( D : pDate; var DL : pDateL );
begin with DL do pDecodeDate( D,YY,MM,DD ) end;


{
   Convert a date in the format YY-MM-DD or YYYY-MM-DD to a datevalue
   The '-' symbol here represents the default date separator character
   in procedure ptd_init()
   Here's how PacdateTime parses date strings.
     Format YY-MM-DD:
       YY = 00..69:  Interpret as 20YY
       YY = 70..99:  Interpret as 19YY
     Format YYYY-MM-DD interpret as presented.
   *** PASSING A DATE EARLIER THAN 1800-01-01 MAY CAUSE INTERESTING THINGS TO HAPPEN.
 }
function pCtoD( CD : string ) : pDate;
begin pCtoD := CtoD( CD )*int64(86400) end;


{
   Convert a character string of '00:00:00'..'23:59:59' to a timevalue
   Here's how PacDateTime parses time strings.
     Format xx:yy is interpreted as     HH:MM:00
     Format xx:yy:zz is interpreted as  HH:MM:SS
     If time under 60 mins, use format  00:MM:SS
   *** NO ERROR CHECKING
}
function pCtoT( CT : string ) : pTime;
var
  H,M,S : byte;
  Frag : string[ 2 ];
  R : pTime;
begin
  if Length( CT ) = 8 then begin
    Frag := LeftStr( CT,2 ); H := AtoI( Frag );
    Frag := Copy( CT,4,2 ); M := AtoI( Frag );
    Frag := RightStr( CT,2 ); S := AtoI( Frag );
  end else begin;
    Frag := LeftStr( CT,2 ); H := AtoI( Frag );
    Frag := RightStr( CT,2 ); M := AtoI( Frag );
    S := 0;
  end;
  pCtoT := ((H*int64(60)+M)*int64(60))+int64(S);
end;

{ Convert time to elapsed minutes since Midnight.  String MUST be full 'HH:MM' }
function pCtoMMM( C : string ) : integer;
var HH,MM : byte;
begin
  HH := AtoI( Copy( C,1,2 ));
  MM := AtoI( Copy( C,4,2 ));
  pCtoMMM := (HH*60)+MM;
end;

{ Convert elapsed minutes since Midnight to Civil Time.  Returns 'HH:MM' }
function pMMMtoC( M : integer ) : string;
var
  S : string;
  HH,MM : byte;
begin
  HH := M DIV 60;
  MM := M MOD 60;
  S := '';
  if HH < 10 then S := '0';
  S := S+IntToStr( HH )+':';
  if MM < 10 then S := S+'0';
  S := S+IntToStr( MM );
  pMMMtoC := S;
end;

procedure pCtoMD( CD : string; var M,D : byte );
var Y : word;
begin pDecodeDate( pCtoD( '2000-'+CD ),Y,M,D ) end;


function pDtoC( D : pDate ) : string;
begin pDtoC := DtoC( D DIV 86400 ) end;


function pTtoC( T : pTime ) : string;
begin
    T := T MOD 86400;
    WriteLn('T = '+N2S(T));
    pTtoC := TtoC( T DIV 3600,(T MOD 3600) DIV 60 )
end;


function pDayOfWeek( D : pDate ) : byte;
begin pDayOfWeek := DayOfWeek( D DIV int64(86400) ) end;


function WeekdayName( D : byte ) : string;
begin WeekdayName := WkD[ D ] end;


function ShortWeekdayName( D : byte ) : string;
begin ShortWeekdayName := Copy( WkD[ D ],1,3 ) end;

function ShortMonthName( M : byte ) : string;
var S : string;
begin
  S := '';
  case M of
    1 : S := 'Jan';
    2 : S := 'Feb';
    3 : S := 'Mar';
    4 : S := 'Apr';
    5 : S := 'May';
    6 : S := 'Jun';
    7 : S := 'Jul';
    8 : S := 'Aug';
    9 : S := 'Sep';
    10 : S := 'Oct';
    11 : S := 'Nov';
    12 : S := 'Dec';
  end;
  ShortMonthName := S;
end;

procedure ptd_init;
begin DateSep := '-' end;

procedure td_init;
begin
   DiM[ 1 ] := 31;                     { Assign days-in-month values for }
   DiM[ 2 ] := 28;                     { the twelve months in the year }
   DiM[ 3 ] := 31;
   DiM[ 4 ] := 30;
   DiM[ 5 ] := 31;
   DiM[ 6 ] := 30;
   DiM[ 7 ] := 31;
   DiM[ 8 ] := 31;
   DiM[ 9 ] := 30;
   DiM[ 10 ] := 31;
   DiM[ 11 ] := 30;
   DiM[ 12 ] := 31;
   DtM[ 1 ] := 0;                      { Assign days-to-month values }
   DtM[ 2 ] := 31;                     { (Number of days prior to the }
   DtM[ 3 ] := 59;                     { first of the month) }
   DtM[ 4 ] := 90;
   DtM[ 5 ] := 120;
   DtM[ 6 ] := 151;
   DtM[ 7 ] := 181;
   DtM[ 8 ] := 212;
   DtM[ 9 ] := 243;
   DtM[ 10 ] := 273;
   DtM[ 11 ] := 304;
   DtM[ 12 ] := 334;
   NoM[ 1 ] := 'January';              { Month names }
   NoM[ 2 ] := 'February';
   NoM[ 3 ] := 'March';
   NoM[ 4 ] := 'April';
   NoM[ 5 ] := 'May';
   NoM[ 6 ] := 'June';
   NoM[ 7 ] := 'July';
   NoM[ 8 ] := 'August';
   NoM[ 9 ] := 'September';
   NoM[ 10 ] := 'October';
   NoM[ 11 ] := 'November';
   NoM[ 12 ] := 'December';
   WkD[ 1 ] := 'Sunday';               { Weekday names }
   WkD[ 2 ] := 'Monday';
   WkD[ 3 ] := 'Tuesday';
   WkD[ 4 ] := 'Wednesday';
   WkD[ 5 ] := 'Thursday';
   WkD[ 6 ] := 'Friday';
   WkD[ 7 ] := 'Saturday';
   ptd_Init;
end;

var Count : byte;

Initialization
   TD_Init;                            { Initialize TIMEDATE }
end.

