{ *************************************************************************

                           EXTENDED STRING LIBRARY

     For Turbo Pascal 5.0

     Commenced 90.03.07
     William C. Demmery

     Version:  93.06.21

  ************************************************************************* }


unit PacString;

{$mode objfpc}{$H+}

interface

uses Math, SysUtils;

//            2016-01-15  2.00 Establishing version 2.0
const SOFTWARE_VERSION = '2.00';

function Replicate( S : string; R : byte ) : string;
function Space( R : byte ) : string;
function Stuff( S : string; Start,Del : byte; Ins : string ) : string;
function LTrim( S : string; C : char ) : string;
function LTrim( S : string ) : string;
function RTrim( S : string; C : char ) : string;
function RTrim( S : string ) : string;
function Trim( S : string; C : char ) : string;
function Trim( S : string ) : string;
function AllTrim( S : string; C : char ) : string;
function AllTrim( S : string ) : string;
function Upper( Lwr : string ) : string;
function Lower( Upr : string ) : string;
function LSet( S : string; L : byte ) : string;
function RSet( S : string; L : byte ) : string;
function CSet( S : string; L : byte ) : string;
function LJust( S : string; L : byte ) : string;
function RJust( S : string; L : byte ) : string;
function CJust( S : string; L : byte ) : string;
function SubStr( S : string; Start,Len : byte ) : string;
function MidStr( S : string; Start,Len : byte ) : string;
function Mid( S : string; Start,Len : byte ) : string;
function Left( S : string; Len : byte ) : string;
function LeftStr( S : string; Len : byte ) : string;
function Right( S : string; Len : byte ) : string;
function RightStr( S : string; Len : byte ) : string;
function Count_Leading( T : char; S : string ) : byte;
function Count_Trailing( T : char; S : string ) : byte;
function IsAlpha( C : char ) : boolean;
function IsNum( C : char ) : boolean;
function IsAlphanum( C : char ) : boolean;
function IsDigit( C : char ) : boolean;
function IsAlphaDigit(C : Char) : Boolean;
function CWeave( S1,S2 : string ) : string;
procedure StrCat( var S1 : string; S2 : string );
function N2S( L : longint ) : string;
function CN2S( L : longint; w : byte ) : string;
function R2S( R : double ) : string;
function CR2S( R : double; W,D : byte ) : string;
function AtoB( S : string ) : byte;
function AtoI( S : string ) : integer;
function AtoL( S : string ) : longint;
function AtoR( S : string ) : double;
function BooToStr(B : boolean; YesValue,NoValue : String) : String;
function BooToStrYN(B : boolean ) : String;
function BooToStrTF(B : boolean) : String;
//function DecToHex( W : longint ) : string;
function DecToHex(W : word; L : String; P : boolean) : string;
function DecToHex(W : word) : String;
function HexToDec(H : String) : longint;
function SToken( C : char; var S : AnsiString ) : AnsiString;
function SToken( S1 : AnsiString; var S2 : AnsiString ) : AnsiString;
function ScaleFileSize( Sz : int64 ) : string;

implementation


//28 uses Misc;


const HexStr : string = '0123456789ABCDEF';


var Code : integer;


{ 
   Generate a string of r repetitions of string S 
}
function Replicate( S : string; R : byte ) : string;
var
   Rep : string;
   Count : byte;
begin
   Rep := '';
   if (R > 0) and (S <> '') then
      for Count := 1 to R do if Length( Rep )+Length( S ) < 256 then
         Rep := Concat( Rep,S );
   Replicate := Rep;
end;



{ 
   Generate a string of r space characters 
}
function Space( R : byte ) : string;
var
  Rep : string;
  Count : byte;
begin
//   Rep[ 0 ] := char(R);
  if R = 0 then Rep := '' else begin
    SetLength( Rep,R );
    FillChar( Rep[ 1 ],R,#32 );
  end;
  Space := Rep;
end;


{
  Insert string ins into string s, at position start,
  optionally first deleting del characters from s.
}
function Stuff( S : string; Start,Del : byte; Ins : string ) : string;
begin
   if Del > 0 then Delete( S,Start,Del );
   if Ins <> '' then
      if Start <> Length( S ) then Insert( S,Ins,Start ) else S := S+Ins;
   Stuff := S;
end;



{
   Trim leading characters from a string
}
function LTrim( S : string; C : char ) : string;
var OK : boolean;
begin
   OK := FALSE;
   while not OK do begin
      if Length( S ) = 0 then OK := TRUE else
         if S[ 1 ] = C then Delete( S,1,1 ) else OK := TRUE;
   end;
   LTrim := S;
end;

{   Trim leading spaces   }
function LTrim( S : string ) : string;
begin LTrim := LTrim( S, ' ' ) end;


{ 
   Trim trailing Characters from a string
}
function RTrim( S : string; C : char ) : string;
var OK : boolean;
begin
   OK := FALSE;
   while not OK do begin
      if Length( S ) = 0 then OK := TRUE else
         if S[ Length( S ) ] = C then Delete( S,Length( S ),1 ) else
            OK := TRUE;
   end;
   RTrim := S;
end;

{   Trim trailing spaces   }
function RTrim( S : string ) : string;
begin RTrim := RTrim( S, ' ' ) end;


{ 
   Trim leading and trailing chars from a string
}
function Trim( S : string; C : char ) : string;
var T : string;
begin
   T := '';
   if Length( S ) > 0 then begin
      T := LTrim( S,C );
      if Length( T ) > 0 then T := RTrim( T,C );
   end;
   Trim := T;
end;

function Trim( S : string ) : string;
begin Trim := Trim( S, ' ' ) end;


{ 
   Synonymous with trim 
}
function AllTrim( S : string; C : char ) : string;
begin   AllTrim := Trim( S, C )   end;


function AllTrim( S : string ) : string;
begin   AllTrim := Trim( S, ' ' )   end;


{ 
   Force all characters within a string to uppercase 
}
function Upper( Lwr : string ) : string;
var Count,Ch : byte;
begin
   if Length( Lwr ) > 0 then
      for Count := 1 to Length( Lwr ) do begin
         Ch := byte(Lwr[ Count ]);
         if (Ch > 96) and (Ch < 123) then
            Lwr[ Count ] := Chr( Ch-32 );
      end;
   Upper := Lwr;
end;


{
   Force all characters within a string to lowercase
}
function Lower( Upr : string ) : string;
var Count,Ch : byte;
begin
   if Length( Upr ) > 0 then
      for Count := 1 to Length( Upr ) do begin
         Ch := byte(Upr[ Count ]);
         if (Ch > 64) and (Ch < 091) then
            Upr[ Count ] := Chr( Ch+32 );
      end;
   Lower := Upr;
end;


{
  Left-set a string:
  - If less than specified length l, pad to right with spaces
  - If greater than l, delete characters from right
}

function LSet( S : string; L : byte ) : string;
begin
   if Length( S ) < L then S := Concat( S,Space( L-Length( S )) );
   if Length( S ) > L then Delete( S,L,Length( S )-L );
   LSet := S;
end;


{ 
   Right-set a string 
}
function RSet( S : string; L : byte ) : string;
begin
   if Length( S ) < L then S := Concat( Space( L-Length( S )),S );
   if Length( S ) > L then Delete( S,1,Length( S )-L );
   RSet := S;
end;


{ 
   Centre-set a string 
}
function CSet( S : string; L : byte ) : string;
var LP,RP : byte;
begin
   if Length( S ) < L then begin
      LP := (L-Length( S )) div 2;
      RP := (L-Length( S ))-LP;
      if LP > 0 then S := Concat( Space( LP ),S );
      S := Concat( S,Space( RP ));
   end;
   if Length( S ) > L then begin
      LP := (Length( S )-L) div 2;
      RP := (L-Length( S ))-LP;
      Delete( S,Succ( Length( S )-RP ),RP );
      if LP > 0 then Delete( S,1,LP );
   end;
   CSet := S;
end;


{
  Left-justify a string:
  - trim leading/trailing spaces
  - perform lset, as above
}
function LJust( S : string; L : byte ) : string;
begin
   if Length( S ) > 0 then S := Ltrim( S );
   LJust := LSet( S,L );
end;


{ 
   Right-justify a string 
}
function RJust( S : string; L : byte ) : string;
begin
   if Length( S ) > 0 then S := RTrim( S );
   RJust := RSet( S,L );
end;

{ 
   Center-justify a string 
}
function CJust( S : string; L : byte ) : string;
begin
   if Length( S ) > 0 then S := Trim( S );
   CJust := CSet( S,L );
end;


{ 
   Get len chars from string s, starting at point start 
}
function SubStr( S : string; Start,Len : byte ) : string;
var SS : string;
begin
   SS := '';
   if Length( S ) >= Start then
      if Length( S ) >= Pred( Start+Len ) then
         SS := Copy( S,Start,Len ) else
         SS := Copy( S,Start,Succ( Length( S )-Start ));
   SubStr := SS;
end;


{ 
   Alternate for SubStr, above 
}
function MidStr( S : string; Start,Len : byte ) : string;
begin MidStr := Copy( S,Start,Len ) end;


{ 
   A second alternate for SubStr, above 
}
function Mid( S : string; Start,Len : byte ) : string;
begin Mid := SubStr( S,Start,Len ) end;


{ 
   Returns len leftmost characters from string S 
}
function Left( S : string; Len : byte ) : string;
var SS : string;
begin
   SS := '';
   if Length( S ) > 0 then
      if Length( S ) >= Len then SS := Copy( S,1,Len ) else SS := S;
   Left := SS;
end;


{ 
   Alternate for Left, above 
}
function LeftStr( S : string; Len : byte ) : string;
begin LeftStr := Left( S,Len ) end;


{ 
   Returns len rightmost characters from string S 
}
function Right( S : string; Len : byte ) : string;
var SS : string;
begin
   if Length( S ) > 0 then
      if Length( S ) >= Len then
         SS := Copy( S,Succ( Length( S )-Len ),Len ) else SS := S;
   Right := SS;
end;


{ 
   Alternate for Right, above 
}
function RightStr( S : string; Len : byte ) : string;
begin RightStr := Right( S,Len ) end;


{ 
   Count the occurrences of the leading character t in string S 
}
function Count_Leading( T : char; S : string ) : byte;
var
   Okay : boolean;
   Count,Counter : byte;
begin
   Okay := False;
   Count := 0;
   Counter := 1;
   if (Length( S ) > 0) and (T <> '' ) then
      while not Okay do begin
         if S[ Counter ] <> T then Okay := True else begin
            Inc( Counter );
            Inc( Count );
            if Counter > Length( S ) then Okay := True;
         end;
      end;
   Count_Leading := Count;
end;


{ 
   Count occurrences of trailing character t in string S 
}
function Count_Trailing( T : char; S : string ) : byte;
var
   Okay : boolean;
   Count,Counter : byte;
begin
   Okay := False;
   Count := 0;
   Counter := Length( S );
   if (Length( S ) > 0) and (T <> '' ) then
      while not Okay do begin
         if S[ Counter ] <> T then Okay := True else begin
            Dec( Counter );
            Inc( Count );
            if Counter = 0 then Okay := True;
         end;
      end;
   Count_Trailing := Count;
end;


{ 
   Returns TRUE if character c is alpha only 
}
function IsAlpha( C : char ) : boolean;
begin
   C := UpCase( C );
   if (C >= 'A') and (C <= 'Z') then IsAlpha := True else IsAlpha := False;
end;


{ 
   Returns TRUE if character c is numeric 
}
function IsNum( C : char ) : boolean;
begin
   if Pos( C,'0123456789.-+' ) <> 0 then IsNum := True else IsNum := False;
end;


{ 
   Returns TRUE if character c is alphanumeric 
}
function IsAlphanum( C : char ) : boolean;
begin
   C := Upcase( C );
   IsAlphanum := (IsAlpha( C )) or (IsNum( C ))
end;

{
   Returns TRUE if character c is a numeral 
}
function IsDigit( C : char ) : boolean;
begin
   if Pos( C,'0123456789' ) <> 0 then IsDigit := True else IsDigit := False;
end;

{
    Returns TRUE if character C is alpha or a digit (compare with AlphaNum())
}
function IsAlphaDigit(C : Char) : Boolean;
begin
    C := UpCase(C);
    IsAlphaDigit := (IsAlpha(C)) or (IsDigit(C))
end;


{
   Returns TRUE if character c is a hex numeral (0..9,A..F)
}
function IsHexNum(C : char) : boolean;
begin
   if Pos(UpCase(C),HexStr) <> 0 then IsHexNum := TRUE else IsHexNum := FALSE;
end;

{
  Weaves two strings together
  - 1st char from s1, then 1st from s2, ...
  - Ie:  cweave( 'xxx','yyy' ) returns 'xyxyxy'
}

function CWeave( S1,S2 : string ) : string;
var
   Res : string;
   Counter : byte;
begin
   Res := '';
   Counter := 1;
   while Counter < Max( Length( S1 ),Length( S2 ) ) do begin
      if Counter <= Length( S1 ) then Res := Res+Copy( S1,Counter,1 );
      if Counter <= Length( S2 ) then Res := Res+Copy( S2,Counter,2 );
      Inc( Counter );
   end;
   CWeave := Res;
end;


{   
   Concatenates the second string to the first   
}
procedure StrCat( var S1 : string; S2 : string );
begin S1 := Concat( S1,S2 ) end;


{   
   Converts an ordinal value to a string   
}
function N2S( L : longint ) : string;
var Res : string;
begin Str( L,Res ); N2S := Res end;


{   
   Converts an ordinal value to a string, with width specification   
}
function CN2S( L : longint; W : byte ) : string;
var Res : string;
begin Str( L:W,Res ); CN2S := Res end;


{   
   Converts a real to a string   
}
function R2S( R : double ) : string;
var Res : string;
begin Str( R,Res ); R2S := Res end;


{   
   Converts a real to a string with width and decimals specifications
}
function CR2S( R : double; W,D : byte ) : string;
var Res : string;
begin Str( R:W:D,Res ); CR2S := Res end;


{   
   Converts a string to a byte value   
}
function AtoB( S : string ) : byte;
var Res : byte;
begin
   Val( Trim( S ),Res,Code );
   AtoB := Res;
end;


{   
   Converts a string to an integer value   
}
function AtoI( S : string ) : integer;
var Res : integer;
begin
   Val( Trim( S ),Res,Code );
   AtoI := Res;
end;


{   
   Converts a string to a longint value   
}
function AtoL( S : string ) : longint;
var Res : longint;
begin
   Val( Trim( S ),Res,Code );
   AtoL := Res;
end;


{   
   Converts a string to a real value   
}
function AtoR( S : string ) : double;
var Res : double;
begin
   Val( Trim( S ),Res,Code );
   AtoR := Res;
end;

{ Converts a boolean value to a custom string representation }
function BooToStr(B : boolean; YesValue,NoValue : String) : String;
var R : String;
begin
    if B then R := YesValue else R := NoValue;
    BooToStr := R;
end;

{ Converts a boolean value to a Yes/No scheme }
function BooToStrYN(B : boolean ) : String;
var R : String;
begin
    R := BooToStr(B,'Yes','No');
    BooToStrYN := R;
end;

{ Converts a boolean vale to a True/False scheme }
function BooToStrTF(B : boolean) : String;
var R : String;
begin
    R := BooToStr(B,'True','False');
    BooToStrTF := R;
end;


//function DecToHex( W : longint ) : string;
//var
//   Res : string;
//   HiW,LoW : word;
// procedure Conv( B : byte );
// var A : byte;
// begin
//    if (Res <> '') OR (B > 0) then begin
//       A := ((B AND 240) DIV 16)+1;
//       Res := Res+HexStr[ A ];
//       A := (B AND 15)+1;
//       Res := Res+HexStr[ A ];
//    end;
// end;
//begin
//   if W = 0 then Res := '00' else begin
//      Res := '';
//      HiW := W SHR 16;
//      Conv( Hi( HiW ));
//      Conv( Lo( HiW ));
//      LoW := W AND word(65535);
//      Conv( Hi( LoW ));
//      Conv( Lo( LoW ));
//   end;
//   DecToHex := Res;
//end;

{
   Convert a word value to a hex representation.
       W:  Value to convert
       L:  Prefix string ('$', '&h')
       P:  Pad with leading zero to 4 chars
}
function DecToHex(W : word; L : String; P : boolean) : string;
var
    R : String;
    Hi,Lo : byte;
procedure Conv( B : byte );
var A : byte;
begin
    A := (B SHR 4)+1;
    R += HexStr[ A ];
    A := (B AND 15)+1;
    R += HexStr[ A ];
end;
begin
    Hi := W SHR 8;
    Lo := W AND 255;
    if Hi = 0 then begin
        if P then Conv(Hi);
    end else Conv(Hi);
    Conv(Lo);
    DecToHex := L+R;
end;

function DecToHex(W : word) : String;
begin DecToHex := DecToHex(W, '$', FALSE) end;

{
    Convert a hex string to a decimal value
        H : Hex string to convert
            (Leading '$' or '&h' will be stripped)
        Returns -1 on error
}
function HexToDec(H : String) : longint;
var
    R : longint;
    P : byte;
    C : char;
begin
    P := 1;
    R := 0;
    H := AllTrim(StringReplace(H,'$','',[rfReplaceAll]));
    if H = '' then R := -1 else begin
        while P <= Length(H) do begin
            if NOT(IsHexNum(H[P])) then begin
                R := -1;
                P := Length(H);
            end else begin
                R := (R*16)+(Pos(H[P],HexStr)-1);
            end;
            P += 1;
        end;
    end;
    HexToDec := R;
end;

{   Search for a separator char, cut the value from the string and return it   }
function SToken( C : char; var S : AnsiString ) : AnsiString;
var
  P : byte;
  R : AnsiString;
begin
  R := '';
  P := Pos( C,S );
  if P <> 0 then begin
    R := Copy( S,1,P-1 );
    Delete( S,1,P );
  end else begin
    R := S;
    S := '';
  end;
  SToken := AllTrim( R );
end;

function SToken( S1 : AnsiString; var S2 : AnsiString ) : AnsiString;
var
  P : byte;
  R : AnsiString;
begin
  R := '';
  P := Pos( S1,S2 );
  if P <> 0 then begin
    R := Copy( S2,1,P-1 );
    Delete( S2,1,(P+Length(S1))-1 );
  end else begin
    R := S2;
    S2 := '';
  end;
  SToken := AllTrim( R );
end;

function ScaleFileSize( Sz : int64 ) : string;
var S : string;
begin
  if Sz < 1000 then S := N2S( Sz )+'b'
  else if Sz < 10000 then S := CR2S( Round( Sz/100 )/10,3,1 )+'K'
  else if Sz < 1000000 then S := N2S( Round( Sz/1000 ))+'K'
  else if Sz < 10000000 then S := CR2S( Round( Sz/100000 )/10,3,1 )+'M'
  else if Sz < 1000000000 then S := N2S( Round( Sz/1000000 ))+'M'
  else if Sz < 10000000000 then S := CR2S( Round( Sz/100000000 )/10,3,1 )+'G'
  else S := N2S( Round( Sz/1000000000 ))+'G';
  ScaleFileSize := S;
end;

end.

