program wxCondition;

{$mode objfpc}{$H+}

uses
  {$IFDEF UNIX}{$IFDEF UseCThreads}
  cthreads,
  {$ENDIF}{$ENDIF}
  Classes, SysUtils, Math,
  PacString, PacDateTime;


CONST
    // 1.00  2017-05-13  First version.
    // 1.02  2017-07-14  Fixed aberrant behaviour when more than one sky condition
    // 1.04  2017-10-17  Restored multiple file handling and recording
    // 1.04b 2017-10-17  Fixed a sorting bug
    // 1.04d 2017-10-31  Fixed inability to deal with negative dewpoints
    // 1.04f 2017-11-06  Fixed inability to calculate dew point based on negative temperatures
    // 1.10  2018-09-24  Added ability to insert pending observations from buffer file. Fixed bug in random-wander generator
    // 1.10a 2018-09-26  Multiple terms in the
    // 1.12  2018-11-14  Added -ps option (skip condition injection and just go inject any manual readings.
    //                   Added help when parameters don't make sense. Changed PutInfo() with switch to control whether injecting value from storage.
    // 2.00  2018-12-12  Added -r option to control whether random-walk factor applied
    //                   Observations limited increased to 1,000 - Linear search, for now
    //                   Random-walk element eliminated
    //                   Multiple Donor sources now supported
    //                   Replacement of any parameter possible with instruction string
    // 2.10 2019-01-13   Implemented [Sprint 2]

    SOFTWARE_VERSION = '2.10';
    SOFTWARE_DATE = '2019-01-13';
    INIL = -30000;


TYPE
    //ReadingRec = record
    //    DateTime : String;
    //    Reading : LongInt;
    //end;
    RRec = record {32}
      RDt : ShortString;
      RCo : ShortString;
      RTe, RBP : ShortString;
      RBPTr : ShortString;
      RVs, RRH, RHx, RWC, RDP, RWn : ShortString;
      RWB, RWG, RAQ : ShortString;
      RPr : ShortString;
      RSun : ShortString;
      RPu : ShortString;
    end;


VAR
    FavourOriginal : Boolean;
    DonorLocCode, SourceLocCode, InstrStr : String;
    ReadingsCount : Word;
    //RootValue : Longint;
    //Readings : array[1..1000] of ReadingRec;
    Readings: array[1..1000] of RRec;
    PendingValues : RRec;


procedure ParseIntoRRec(var R : RRec; S : String; DT : Boolean); forward;
function ParseFromRRec(var R : RRec) : String; forward;


procedure InitRRec(var R : RRec);
begin
    with R do begin
        RDt := '';
        RCo := '';
        RTe := '';
        RBP := '';
        RBPTR := 'U';
        RVs := '';
        RRH := '';
        RHx := '';
        RWC := '';
        RDP := '';
        RWn := '';
        RWB := '';
        RWG := '';
        RAQ := '';
        RPr := '';
        RSun := '';
        RPu := 'm';
    end;
end;


procedure Help();
begin
    WriteLn();
    WriteLn('wxCondition v'+SOFTWARE_VERSION+'; '+SOFTWARE_DATE);
    WriteLn('Syntax:');
    WriteLn();
    //WriteLn('  wxCondition -(g|p} {LocCode};');
    //WriteLn('  e.g.:');
    WriteLn('    wxCondition -g {DonorLocCode}  = Capture latest readings from donor');
    WriteLn('    wxCondition -ps CTO  = Skip processing; only inject manual readings');
    WriteLn('    wxCondition -p [-pr {pref}] {SourceLocCode} {DonorLocCode} {InstrString}  = Process normally');
    WriteLn('      -ps = Skip value substitutions and only add manual observations)');
    WriteLn('      -pr A = favour the donor values; O = favour the source values (default)');
    WriteLn('      {InstrString} = LabelInst#Label2Inst2#Label3Inst3#...');
    WriteLn('        Label is a datum label; i.e. Te, RH, BP, DP, Hx, WC, Wn, WB, WG, Pr, Vs, Su)');
    WriteLn('        Inst is one character, optionally followed by an argument:');
    WriteLn('          M = Replace source value with donor value, if source value missing');
    WriteLn('          U = Replace source value with donor value, unconditionally');
    WriteLn('          H = Replace source value with donor value, if donor value is larger');
    WriteLn('          L = Replace source value with donor value, if donor value is smaller');
    WriteLn('          E{arg} = Replace source value with donor value, if source value is equal to {arg}');
    WriteLn('          G{arg} = Replace source value with donor value, if source value is greater than {arg}');
    WriteLn('          S{arg} = Replace source value with donor value, if source value is less than {arg}');
    WriteLn('          A = Replace source value with average of (source value + donor value)');
    WriteLn('          P = Process per preferences and nothing further.');
    WriteLn('  Examples:');
    WriteLn('    wxCondition -g YOW');
    WriteLn('    wxCondition -ps CTO');
    WriteLn('    wxCondition -p -pr A CTO YOW TeM#RHU');
    WriteLn();
end;


procedure CalcDewPoint(Te,RH : double; VAR DPV : longint);
begin
    WriteLn('Te = '+FloatToStr(Te)+'; RH = '+FloatToStr(RH));
    if (Te <> INil) AND (RH <> INil) then begin
        DPV := Round((Power(RH/100,0.125)*(112+(0.9*(Te)))+(0.1*Te)-112)*10);
    end else DPV := INil;
end;


procedure ReadIni();
var
    X : Word;
    InputFile : Textfile;
    FileName, InputLine : String;
    RR : RRec;
begin
    FileName := 'wxCondition-'+DonorLocCode+'.ini';
    if FileExists(FileName) then begin
        AssignFile(InputFile, FileName);
        Reset(InputFile);
        //ReadLn(InputFile, InputLine);
        //RootValue := StrToInt(InputLine);
        //ReadLn(InputFile, InputLine);
        //RandValue := StrToFloat(InputLine);
        ReadLn(InputFile, ReadingsCount);
        if ReadingsCount > 0 then begin
            for X := 1 to ReadingsCount do begin
                 ReadLn(InputFile, InputLine);
                 InitRRec(RR);
                 ParseIntoRRec(RR, InputLine, TRUE);
                 Readings[X] := RR;
                 //Readings[X].RDt := SToken(',', InputLine);
                 //Readings[X].Reading := AToB(InputLine);
                 //WriteLn('Date = '+Readings[X].DateTime+'; Reading = '+N2S(Readings[X].Reading));
            end;
        end;
        CloseFile(InputFile);
    end;
end;


procedure InsertInfoOrdered(RR : RRec);
var
    OK, GotIP : Boolean;
    P, X : Word;
begin
    if ReadingsCount = 0 then begin
        Inc(ReadingsCount);
        Readings[1] := RR;
    end else begin
        OK := FALSE;
        GotIP := FALSE;
        P := 1;
        repeat
            // If found, update the reading value
            if Readings[P].RDT = RR.RDT then begin
                OK := TRUE;
                Readings[P] := RR;
            // If larger, note the insertion point
            end else if RR.RDT > Readings[P].RDT then begin
                OK := TRUE;
                GotIP := TRUE;
            end else begin
                Inc(P);
                if P > ReadingsCount then begin
                    if P < 1001 then begin
                        GotIP := TRUE;
                    end;
                    OK := TRUE;
                end;
            end;
        until OK;
        if GotIP then begin
            if P > ReadingsCount then begin
                Inc(ReadingsCount);
                Readings[P] := RR;
            end else begin
                X := ReadingsCount+1;
                while X > P do begin
                    Readings[X] := Readings[X-1];
                    Dec(X);
                end;
                Readings[P] := RR;
                Inc(ReadingsCount);
                if ReadingsCount > 1000 then ReadingsCount := 1000;
            end;
        end;
    end;
end;


(* Change this routine significantly.
   - Calculate new value but do not apply
   *)
procedure GetInfo(LocCode, FileName : String);
var
    InFile, OutFile : TextFile;
    InputLine, OutputLine, Param : String;
    X : Word;
    Trend : Char;
    Date : String;
    RR : RRec;
    RV : Double;
begin
    OutputLine := '';
    //RootValue := 0;
    WriteLn('FileName = '+FileName);
    if FileExists(FileName) then begin
        AssignFile(InFile, FileName);
        Reset(InFile);
        if NOT Eof(InFile) then begin
            ReadLn(InFile, InputLine);
            ParseIntoRRec(RR, InputLine, TRUE);
            //Date := SToken(',', InputLine);
            //WriteLn('Date = '+Date);
            //for X := 1 to 3 do begin
            //    Param := SToken(',', InputLine);
            //end;
            //Param := SToken(',', InputLine);
            //Trend := Param[1];
            //Param := SToken(',', InputLine);
            //// Target for reading:
            //Param := SToken(',', InputLine);
            //RootValue := StrToInt(Param);
            //// Done with it
            //CloseFile(InFile);
            //
            ////WriteLn('RV = '+FloatToSTr(RV)+'; RandValue = '+FloatToStr(RandValue));
            ////RootValue := Round(RootValue+RandValue);
            //with RR do begin
            //    DateTime := Date;
            //    Reading := RootValue;
            //end;
            InsertInfoOrdered(RR);
        end else begin
            WriteLn('Source file empty');
        end;
    end else begin
        WriteLn('Source File not found');
    end;
    // Rename the file
    RenameFile(FileName, FileName+'t');
    // Rewrite the .ini file
    FileName := 'wxCondition-'+LocCode+'.ini';
    AssignFile(OutFile, FileName);
    ReWrite(OutFile);
    //WriteLn(OutFile, IntToStr(RootValue));
    //WriteLn(OutFile, FloatToStr(RandValue));
    WriteLn(OutFile, N2S(ReadingsCount));
    if ReadingsCount > 0 then begin
        for X := 1 to ReadingsCount do begin
            WriteLn(OutFile, ParseFromRRec(Readings[X]));
        end;
    end;
    CloseFile(OutFile);

end;


function FindReading(Date : String) : Byte;
var
    R, P : Word;
    OK : Boolean;
begin
    R := 0;
    OK := FALSE;
    if ReadingsCount > 0 then begin
        WriteLn('RC > 0');
        P := 1;
        while NOT OK do begin
            if Date >= Readings[P].RDT then begin
                WriteLn('We have a winner at position '+N2S(P));
                R := P;
                OK := TRUE;
            end else begin
                Inc(P);
                if P > ReadingsCount then begin
                    WriteLn('Well, no winner');
                    OK := TRUE;
                end;
            end;
        end;
    end;
    FindReading := R;
end;


{ Parse a line of observations into an RRec
    R : the RRec to parse into
    S : The Source string with observations
    DT : whether the source string contains date and time
    }
procedure ParseIntoRRec(var R : RRec; S : String; DT : Boolean);
var Param : String;
begin
    with R do begin
        if DT then begin
            Param := SToken(',', S);
            RDT := Param;
        end;
        Param := SToken('",', S);
        RCo := Param+'"';
        Param := SToken(',', S);
        RTe := Param;
        Param :=  SToken(',', S);
        RBP := Param;
        Param :=  SToken(',', S);
        RBPTr := Param[1];
        Param :=  SToken(',', S);
        RDP := Param;
        Param :=  SToken(',', S);
        RRH := Param;
        Param :=  SToken(',', S);
        RVs := Param;
        Param :=  SToken(',', S);
        RWB := Param;
        Param :=  SToken(',', S);
        RWn := Param;
        Param :=  SToken(',', S);
        RWG := Param;
        Param :=  SToken(',', S);
        RHx := Param;
        Param :=  SToken(',', S);
        RWC := Param;
        Param :=  SToken(',', S);
        RAQ := Param;
        Param :=  SToken(',', S);
        RPr := Param;
        Param :=  SToken(',', S);
        RSun := Param;
        Param :=  S;
        RPu := Param;
    end;
end;


function eStrToInt(V : String) : LongInt;
var
    RV : LongInt;
begin
    RV := 0;
    if (V = 'nil') OR (V = '-30000') then RV := INIL
      else RV := StrToInt(V);
    Result := RV;
end;


(*
{ Revamp this routine to accommodate the whole Readings record
  Copy to similar routines for the other data
  Insert the saved humidity value into a readings record.
     R : The readings record (must have valid date and time)
     }
procedure InsertRH(var R : RRec);
var
    X : Word;
    Te, RValue, DP, RecNo : LongInt;
    RV : Double;
    OutFile : Textfile;
    RR : RRec;
    FileName : String;
begin
    Te := eStrToInt(R.RTe);
    DP := eStrToInt(R.RDP);
    RecNo := FindReading(R.RDT);
    if RecNo > 0 then begin
        RR := Readings[RecNo];
        RValue := StrToInt(RR.RRH);
        R.RRH := IntToStr(RValue);
        if ((Te <> INIL) AND (RValue <> INIL) AND (DP <> INIL)) then begin
            CalcDewPoint((Te/10), RValue, DP);
            R.RDP := IntToStr(DP);
        end;
    end;
end;
*)


procedure GetPending(LocCode : String; var Obs : RRec);
var
    PendObs : RRec;
    FileName, InputLine : String;
    InputFile : TextFile;
begin
    // Check for pending observations file.
    FileName := LocCode+'-mccpending.txt';
    if FileExists(FileName) then begin
        // If found:
        // Open file, read in observations and store them to PendObs
        AssignFile(InputFile, FileName);
        Reset(InputFile);
        if NOT EoF(InputFile) then begin
            ReadLn(InputFile, Inputline);
            InitRRec(PendObs);
            ParseIntoRRec(PendObs, InputLine, FALSE);
            // Replace any readings in Obs with any valid readings in PendObs
            if PendObs.RCo <> '""' then Obs.RCo := PendObs.RCo;
            if PendObs.RTe <> 'nil' then Obs.RTe := PendObs.RTe;
            if PendObs.RBP <> 'nil' then Obs.RBP := PendObs.RBP;
            if PendObs.RBPTr <> 'nil' then Obs.RBPTr := PendObs.RBPTr;
            if PendObs.RDP <> 'nil' then Obs.RDP := PendObs.RDP;
            if PendObs.RRH <> 'nil' then Obs.RRH := PendObs.RRH;
            if PendObs.RVs <> 'nil' then Obs.RVs := PendObs.RVs;
            if PendObs.RWB <> 'nil' then Obs.RWB := PendObs.RWB;
            if PendObs.RWn <> 'nil' then Obs.RWn := PendObs.RWn;
            if PendObs.RWG <> 'nil' then Obs.RWG := PendObs.RWG;
            if PendObs.RHx <> 'nil' then Obs.RHx := PendObs.RHx;
            if PendObs.RWC <> 'nil' then Obs.RWC := PendObs.RWC;
            if PendObs.RAQ <> 'nil' then Obs.RAQ := PendObs.RAQ;
            if PendObs.RPr <> 'nil' then Obs.RPr := PendObs.RPr;
            if PendObs.RSun <> 'nil' then Obs.RSun := PendObs.RSun;
            if PendObs.RPu <> 'nil' then Obs.RPu := PendObs.RPu;
        end;
        // Close file
        CloseFile(InputFile);
        // Delete file
        DeleteFile(FileName);
    end;
end;


{ Parse from an RRec into an output string }
function ParseFromRRec(var R : RRec) : String;
var
    RV : String;
begin
    with R do begin
        RV := RDT+','+RCo+','+RTe+','+RBP+','+RBPTr+','+RDP+','+RRH+','+RVs+','+RWB+','+RWn+','+RWG+','+RHx+','+RWC+','+RAQ+','+RPr+','+RSun+','+RPu;
    end;
    Result := RV;
end;


function ProcessDatum(var SrcData, DonorData, Instruction, Argument : ShortString) : Boolean;
var
    R : Boolean;
begin
    R := FALSE;
    if NOT FavourOriginal then begin
        SrcData := DonorData;
    end;
    // M for Missing
    if Instruction = 'M' then begin
        if SrcData = 'nil' then begin
            SrcData := DonorData;
            R := TRUE;
        end;
    // U for Unconditionally
    end else if Instruction = 'U' then begin
        SrcData := DonorData;
        R := TRUE;
    // H for Higher value
    end else if Instruction = 'H' then begin
        if DonorData <> 'nil' then begin
            if SrcData = 'nil' then SrcData := DonorData else begin
                if StrToInt(DonorData) > StrToInt(SrcData) then begin
                    SrcData := DonorData;
                    R := TRUE;
                end;
            end;
        end;
    // L for Lower value
    end else if Instruction = 'L' then begin
        if DonorData <> 'nil' then begin
            if SrcData = 'nil' then SrcData := DonorData else begin
                if StrToInt(DonorData) < StrToInt(SrcData) then begin
                    SrcData := DonorData;
                    R := TRUE;
                end;
            end;
        end;
    // E for Equals
    end else if Instruction = 'E' then begin
        if DonorData <> 'nil' then begin
            if SrcData <> 'nil' then begin
                if StrToInt(SrcData) = StrToInt(Argument) then begin
                    SrcData := DonorData;
                    R := TRUE;
                end;
            end;
        end;
    // G for Greater
    end else if Instruction = 'G' then begin
        if DonorData <> 'nil' then begin
            if SrcData <> 'nil' then begin
                if StrToInt(SrcData) > StrToInt(Argument) then begin
                    SrcData := DonorData;
                    R := TRUE;
                end;
            end;
        end;
    // S for leSs
    end else if Instruction = 'S' then begin
        if DonorData <> 'nil' then begin
            if SrcData <> 'nil' then begin
                if StrToInt(SrcData) < StrToInt(Argument) then begin
                    SrcData := DonorData;
                    R := TRUE;
                end;
            end;
        end;
    // A for Average
    end else if Instruction = 'A' then begin
        WriteLn('A for Average');
        if DonorData <> 'nil' then begin
            if SrcData <> 'nil' then begin
                WriteLn('SrcData = '+SrcData+'; DonorData = '+DonorData);
                WriteLn('Averaging...');
                SrcData :=  IntToStr(Trunc((StrToInt(SrcData)+StrToInt(DonorData))/2));
                WriteLn('SrcData = '+SrcData);
                R := TRUE;
            end;
        end;
    end;
    Result := R;
end;


function ProcessDatum(var SrcData1, SrcData2, DonorData1, DonorData2, Instruction, Argument : shortString) : Boolean;
var
    R : Boolean;
begin
    R := FALSE;
    WriteLn('ProcessDatum(); Instruction = '+Instruction);
    if NOT FavourOriginal then begin
        SrcData1 := DonorData1;
        SrcData2 := DonorData2;
        R := TRUE;
    end;
    // M for Missing
    if Instruction = 'M' then begin
        if SrcData1 = 'nil' then begin
            SrcData1 := DonorData1;
            SrcData2 := DonorData2;
            R := TRUE;
        end;
    // U for Unconditionally
    end else if Instruction = 'U' then begin
        SrcData1 := DonorData1;
        SrcData2 := DonorData2;
        R := TRUE;
    // H for Higher value
    end else if Instruction = 'H' then begin
        if DonorData1 <> 'nil' then begin
            if SrcData1 = 'nil' then begin
               SrcData1 := DonorData1;
               SrcData2 := DonorData2;
               R := TRUE;
            end else begin
                if StrToInt(DonorData1) > StrToInt(SrcData1) then begin
                    SrcData1 := DonorData1;
                    SrcData2 := DonorData2;
                    R := TRUE;
                end;
            end;
        end;
    // L for Lower value
    end else if Instruction = 'L' then begin
        if DonorData1 <> 'nil' then begin
            if SrcData1 = 'nil' then begin
                SrcData1 := DonorData1;
                SrcData2 := DonorData2;
                R := TRUE;
            end else begin
                if StrToInt(DonorData1) < StrToInt(SrcData1) then begin
                    SrcData1 := DonorData1;
                    SrcData2 := DonorData2;
                    R := TRUE;
                end;
            end;
        end;
    // E for Equals
    end else if Instruction = 'E' then begin
        if DonorData1 <> 'nil' then begin
            if SrcData1 <> 'nil' then begin
                if StrToInt(SrcData1) = StrToInt(Argument) then begin
                    SrcData1 := DonorData1;
                    SrcData2 := DonorData2;
                    R := TRUE;
                end;
            end;
        end;
    // G for Greater
    end else if Instruction = 'G' then begin
        if DonorData1 <> 'nil' then begin
            if SrcData1 <> 'nil' then begin
                if StrToInt(SrcData1) > StrToInt(Argument) then begin
                    SrcData1 := DonorData1;
                    SrcData2 := DonorData2;
                    R := TRUE;
                end;
            end;
        end;
    // S for leSs
    end else if Instruction = 'S' then begin
        if DonorData1 <> 'nil' then begin
            if SrcData1 <> 'nil' then begin
                if StrToInt(SrcData1) < StrToInt(Argument) then begin
                    SrcData1 := DonorData1;
                    SrcData2 := DonorData2;
                    R := TRUE;
                end;
            end;
        end;
    // A for Average
    end else if Instruction = 'A' then begin
        WriteLn('A for Average');
        if DonorData1 <> 'nil' then begin
            if SrcData1 <> 'nil' then begin
                WriteLn('Averaging...');
                SrcData1 :=  IntToStr(Trunc((StrToInt(SrcData1)+StrToInt(DonorData1))/2));
                R := TRUE;
            end;
            // Operands 2 are not averaged but left as-is
        end;
    end;
    Result := R;
end;


function CalcHumidex(T, DP : longint) : longint;
var
    R : longint;
    A,B,C,D : double;
begin
    R := INil;
    if (T >= 200) AND (DP <> INil) then begin
        A := (1/273.16)-(1/((DP/10)+273.16));
        B := Exp(5417.7530*A);
        C := (6.11*B)-10;
        D := (T/10)+(0.5555*C);
        R := Max(Round(D*10),T);
    end;
    if R < T then R := INil;
    CalcHumidex := R;
end;


function WindChillIndex( T : longint; V : longint ) : longint;
var
    R : longint;
    R1, R2, R3, D : double;
    T0,V0 : double;
begin
    R := INil;
    //Logger.Trace('T = '+N2S(T)+', V = '+N2S(V));
    if (T <> INil) AND (V <> INil) then begin
        if (T <= 100) AND (V > 48) then begin
            T0 := T/10;
            V0 := V;
            R1 := 0.6215*T0;
            //logger.Trace('R1 = '+R2S(R1));
            R2 := 11.37*Power(V0, 0.16 );
            //logger.Trace('R2 = '+R2S(R2));
            R3 := 0.3965*T0*Power( V0, 0.16 );
            //logger.Trace('R3 = '+R2S(R3));
            D := 13.12+R1-R2+R3;
            //logger.Trace('D = '+R2S(D));
            R := Round( D*10 );
            if R > -50 then begin
                R := INil;
                if T <= -50 then begin
                    R := T;
                end;
            end;
        end;
    end;
    WindChillIndex := R;
end;


{ Change this routine to accommodate:
  - Instruction String processing
  }
procedure PutInfo(LocCode, FileName : String; Inject : Boolean);
var
    Error : Boolean;
    InputFile, OutputFile : TextFile;
    Datum, Instruction, Argument : ShortString;
    InputLine, OutputLine, Param, S : String;
    X, Idx, DIdx : Word;
    Te, RH, DP : Double;
    FRes : LongInt;
    Date : String;
    RValue : LongInt;
    Obs, Donor : RRec;
begin
    Error := FALSE;
    WriteLn('Instruction String = '+InstrStr);
    //WriteLn('PutInfo()');
    if FileExists(FileName) then begin
        RValue := INIL;
        DP := INIL;
        // Read Obs file
        OutputLine := '';
        AssignFile(InputFile, FileName);
        Reset(InputFile);
        if NOT Eof(Inputfile) then begin
            ReadLn(InputFile, InputLine);
            ParseIntoRRec(Obs, InputLine, TRUE);
            DIdx := FindReading(Obs.RDT);
            if DIdx > 0 then begin
                Donor := Readings[DIdx];
                if Inject then begin
                    X := 1;
                    if (Length(InstrStr) = 0) (* OR (Length(InstrStr) MOD 5 <> 4) *) then begin
                        Error := TRUE;
                        WriteLn('Don''t understand instruction string-1');
                    end else begin
                        while X <= Length(InstrStr) do begin
                            Datum := '';
                            Instruction := '';
                            Datum := InstrStr[X];
                            Inc(X);
                            Datum += InstrStr[X];
                            Inc(X);
                            //WriteLn('Datum = '+Datum);
                            // Skip the instruction separator
                            // Commenting out the following line 2019-01-10, to remove separator character between operand and opcode
                            //Inc(X);
                            Instruction := InstrStr[X];
                            Inc(X);
                            if Pos('|'+Datum+'|', '|Te|BP|RH|DP|Hx|WC|Wn|WB|WG|Pr|Su|Vs|') = 0 then begin
                                WriteLn('Don''t understand instruction string-2');
                                Error := TRUE;
                                X := Length(InstrStr)+1;
                            end else begin
                                if Pos(Instruction, 'MUHLPEGSA') = 0 then begin
                                    WriteLn('Don''t understand instruction string-3');
                                    X := Length(InstrStr)+1;
                                    Error := TRUE;
                                end else begin
                                    // If an argument follows the instruction, grab it
                                    Argument := '';
                                    while Pos(InstrStr[X], '0123456789') > 0 do begin
                                        Argument := Argument+InstrStr[X];
                                        Inc(X);
                                    end;
                                    if Datum = 'Te' then begin
                                        if ProcessDatum(Obs.RTe, Donor.RTe, Instruction, Argument) then begin
                                            DP := 0;
                                            if Obs.RTe <> 'nil' then begin
                                                Te := StrToFloat(Obs.RTe)/10.0;
                                                WriteLn('Te = '+FloatToStr(Te));
                                                if Obs.RRH <> 'nil' then begin
                                                    RH := StrToFloat(Obs.RRH);
                                                    CalcDewPoint(Te, RH, FRes);
                                                    if FRes = -30000 then Obs.RDP := 'nil' else begin
                                                        Obs.RDP := IntToStr(FRes);
                                                        FRes := CalcHumidex(StrToInt(Obs.RTe), StrToInt(Obs.RDP));
                                                        if Fres = INIL then Obs.RHx := 'nil' else Obs.RHx := IntToStr(FRes);
                                                    end;
                                                end;
                                                if Obs.RWn <> 'nil' then begin
                                                    FRes := WindChillIndex(StrToInt(Obs.RTe), StrToInt(Obs.RWn));
                                                    if FRes = INIL then Obs.RWC := 'nil' else Obs.RWC := IntToStr(FRes);
                                                end;
                                            end;
                                        end;
                                    end else if Datum = 'BP' then begin
                                        ProcessDatum(Obs.RBP, Obs.RBPTr, Donor.RBP, Donor.RBPTr, Instruction, Argument);
                                    end else if Datum = 'RH' then begin
                                        WriteLn('RH');
                                        WriteLn('Instruction = '+Instruction+'; Argument = '+Argument);
                                        if ProcessDatum(Obs.RRH, Donor.RRH, Instruction, Argument) then begin
                                            DP := 0;
                                            if Obs.RTe <> 'nil' then begin
                                                Te := StrToFloat(Obs.RTe)/10.0;
                                                if Obs.RRH <> 'nil' then begin
                                                    RH := StrToFloat(Obs.RRH);
                                                    CalcDewPoint(Te, RH, FRes);
                                                    if FRes = -30000 then Obs.RDP := 'nil' else begin
                                                        Obs.RDP := IntToStr(FRes);
                                                        FRes := CalcHumidex(StrToInt(Obs.RTe), StrToInt(Obs.RDP));
                                                        if Fres = INIL then Obs.RHx := 'nil' else Obs.RHx := IntToStr(FRes);
                                                    end;
                                                end;
                                            end;
                                        end else WriteLn('Error Processing Datum');
                                    end else if Datum = 'DP' then begin
                                        if ProcessDatum(Obs.RDP, Donor.RDP, Instruction, Argument) then begin
                                            if Obs.RTe <> 'nil' then begin
                                                FRes := CalcHumidex(StrToInt(Obs.RTe), StrToInt(Obs.RDP));
                                                if Fres = INIL then Obs.RHx := 'nil' else Obs.RHx := IntToStr(FRes);
                                            end;
                                        end;
                                    end else if Datum = 'Hx' then begin
                                        ProcessDatum(Obs.RHx, Donor.RHx, Instruction, Argument);
                                    end else if Datum = 'WC' then begin
                                        ProcessDatum(Obs.RWC, Donor.RWC, Instruction, Argument);
                                    end else if Datum = 'Wn' then begin
                                        ProcessDatum(Obs.RWn, Donor.RWn, Instruction, Argument);
                                        if Obs.RTe <> 'nil' then begin
                                            if Obs.RWn <> 'nil' then begin
                                                FRes := WindChillIndex(StrToInt(Obs.RTe), StrToInt(Obs.RWn));
                                                if FRes = INIL then Obs.RWC := 'nil' else Obs.RWC := IntToStr(FRes);
                                            end;
                                        end;
                                    end else if Datum = 'WB' then begin
                                        ProcessDatum(Obs.RWB, Donor.RWB, Instruction, Argument);
                                    end else if Datum = 'WG' then begin
                                        ProcessDatum(Obs.RWG, Donor.RWG, Instruction, Argument);
                                    end else if Datum = 'Pr' then begin
                                        ProcessDatum(Obs.RPr, Obs.RPU, Donor.RPr, Donor.RPU, Instruction, Argument);
                                    end else if Datum = 'Su' then begin
                                        ProcessDatum(Obs.RSun, Donor.RSun, Instruction, Argument);
                                    end else if Datum = 'Vs' then begin
                                        ProcessDatum(Obs.RVs, Donor.RVs, Instruction, Argument);
                                    end else begin
                                        WriteLn('Don''t understand instruction string-4');
                                        Error := TRUE;
                                        X := Length(InstrStr)+1;
                                    end;
                                    // Skip the instruction separator
                                    Inc(X);
                                    //WriteLn('X = '+IntToStr(X));
                                end;
                            end;
                        end;
                    end;
                end;
            end;
        end;
        CloseFile(InputFile);
        if NOT Error then begin
            DeleteFile(FileName);
            GetPending(LocCode, Obs);
            // Write it to the obs file
            AssignFile(OutputFile, FileName+'t');
            ReWrite(OutputFile);
            OutputLine := ParseFromRRec(Obs);
            WriteLn(OutputFile, OutputLine);
            CloseFile(OutputFile);
        end;
    end;
end;


var
    Error, OK: Boolean;
    X : Byte;
    FileName, Param : String;
    SR : TSearchRec;
    ErrorCode : LongInt;
begin
    Error := FALSE;
    //Randomize;
    FavourOriginal := TRUE;
    //ApplyRand := FALSE;
    //ManualOnly := FALSE;
    ReadingsCount := 0;
    if ParamCount() < 2 then begin
        Help();
    end else begin
        X := 1;
        Param := ParamStr(X);
        Inc(X);
        if Param[1] = '-' then Delete(Param, 1, 1);
        if Length(Param) = 0 then begin
            WriteLn('Parameter missing');
            Error := TRUE;
        end else begin
            if Param = 'p' then begin
                WriteLn('Put');
                OK := TRUE;
                while OK do begin
                    if X > ParamCount then begin
                        WriteLn('Out of parameters');
                        Error := TRUE;
                        OK := FALSE;
                    end else begin
                        Param := ParamStr(X);
                        Inc(X);
                        WriteLn('Param = '+Param);
                        if Param[1] = '-' then begin
                            WriteLn('Flag');
                            Delete(Param, 1, 1);
                            if Param = 'pr' then begin
                                WriteLn('Favourite selected');
                                if X > ParamCount then begin
                                    WriteLn('Out of parameters');
                                    Error := TRUE;
                                    OK := FALSE;
                                end else begin
                                    Param := ParamStr(X);
                                    Inc(X);
                                    if Param = 'O' then FavourOriginal := TRUE
                                    else if Param <> 'A' then begin
                                        Error := TRUE;
                                        OK := FALSE;
                                    end else begin
                                        FavourOriginal := FALSE;
                                    end;
                                end;
                            end else begin
                                WriteLn('Invalid parameter '+Param);
                                Error := TRUE;
                                OK := FALSE;
                            end;
                        end else begin
                           WriteLn('Parameter not a flag');
                           OK := FALSE;
                        end;
                    end;
                end;
                WriteLn('Got here.');
                if NOT Error then begin
                    WriteLn('No error at this point');
                    //Param := ParamStr(X);
                    //Inc(X);
                    //if X > ParamCount then Error := TRUE else begin
                        WriteLn('SourceLocCode = '+Param);
                        SourceLocCode := Param;
                        Param := ParamStr(X);
                        Inc(X);
                        DonorLocCode := Param;
                        WriteLn('DonorLocCode = '+Param);
                        if X > ParamCount then Error := TRUE else begin
                            Param := ParamStr(X);
                            Inc(X);
                            InstrStr := Param;
                            ReadIni();
                            ErrorCode := FindFirst(SourceLocCode+'-Observations*.tx', faAnyFile, SR);
                            if ErrorCode = 0 then begin
                                repeat
                                    FileName := SR.Name;
                                    PutInfo(SourceLocCode, FileName, TRUE);
                                    ErrorCode := FindNext(SR);
                                until ErrorCode <> 0;
                            end else WriteLn('File not found.');
                            FindClose(SR);
                        end;
                    //end;
                end;
            end else if Param = 'g' then begin
                Param := ParamStr(X);
                Inc(X);
                DonorLocCode := Param;
                ReadIni();
                ErrorCode := FindFirst(DonorLocCode+'-Observations*.tx', faAnyFile, SR);
                if ErrorCode = 0 then begin
                    repeat
                        FileName := SR.Name;
                        GetInfo(DonorLocCode, FileName);
                        ErrorCode := FindNext(SR);
                    until ErrorCode <> 0;
                end else WriteLn('File not found.');
                FindClose(SR);
            end else if Param = 'ps' then begin
                Param := ParamStr(X);
                Inc(X);
                SourceLocCode := Param;
                ErrorCode := FindFirst(SourceLocCode+'-Observations*.tx', faAnyFile, SR);
                if ErrorCode = 0 then begin
                    repeat
                        FileName := SR.Name;
                        PutInfo(SourceLocCode, FileName, FALSE);
                        ErrorCode := FindNext(SR);
                    until ErrorCode <> 0;
                end else WriteLn('File not found.');
                FindClose(SR);
            end;
        end;
        if Error then Help();
        //Inc(X);
        //
        //if (Mode = 'g') OR ((Mode = 'p') OR (Mode = 'ps')) then begin
        //    // Find a file beginning with the locCode given
        //    ErrorCode := FindFirst(LocCode+'-Observations*.tx', faAnyFile, SR);
        //    if ErrorCode = 0 then begin
        //        repeat
        //            FileName := SR.Name;
        //            if Mode = 'g' then GetInfo(LocCode, FileName) else begin
        //                if Mode = 'p' then PutInfo(LocCode, FileName, TRUE) else PutInfo(LocCode, FileName, FALSE);
        //            end;
        //            ErrorCode := FindNext(SR);
        //        until ErrorCode <> 0;
        //    end else WriteLn('File not found.');
        //    FindClose(SR);
        //end else Help();
    end;
end.

